package cc.smarnet.firmwareupdate.firmware;

import android.os.Handler;
import android.os.Looper;
import android.os.SystemClock;

import androidx.annotation.NonNull;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import cc.smarnet.firmwareupdate.bluetooth.DeviceConnFactoryManager;

public class Firmware {

    /**
     * 固件CRC校验
     */
    private static long[] crcTbl = new long[]{
            0xedb88320L, 0xf00f9344L, 0xd6d6a3e8L, 0xcb61b38cL,
            0x00000000L, 0x1db71064L, 0x3b6e20c8L, 0x26d930acL,
            0x76dc4190L, 0x6b6b51f4L, 0x4db26158L, 0x5005713cL,
            0x9b64c2b0L, 0x86d3d2d4L, 0xa00ae278L, 0xbdbdf21cL,
    };

    private Handler handler = new Handler(Looper.getMainLooper());

    /**
     * 更新固件
     *
     * @param file    打印机固件
     * @param deviceConnFactoryManager   io
     * @param firmwareResult 更新回调
     */
    public void updateFirmware(@NonNull File file, @NonNull final DeviceConnFactoryManager deviceConnFactoryManager, @NonNull final FirmwareResult firmwareResult) {
        try {
            final byte[] firmwareBin = convertFileToByte(file);
            //校验固件
            if (verificationBin(firmwareBin)) {
                firmwareResult.startUpdate();
                //固件信息
                byte[] firmwareUpdateInfo = Arrays.copyOfRange(firmwareBin, firmwareBin.length - 72, firmwareBin.length - 8);
                byte[] firmwareUpdateSize = convertIntToByteArrays(firmwareBin.length);
                byte[] updateCommand = new byte[]{0x0a, 0x1f, 0x1b, 0x1f, 0x1f, 0x1b, 0x1f, 0x1f, 0x1b, 0x1f, 0x02, 0x03, 0x04};
                //固件更新命令
                byte[] firmwareUpdateCommand = mergeArrays(updateCommand, firmwareUpdateSize);
                List<byte[]> list = getListByteArray(mergeArrays(firmwareUpdateCommand, firmwareUpdateInfo, firmwareBin), 1024);
                int progress;
                for (int i = 0; i < list.size(); i++) {
                    deviceConnFactoryManager.write(list.get(i));
                    progress = (int) ((float) i / (float) list.size() * 100);
                    firmwareResult.progress(progress);
                    SystemClock.sleep(200);
                }
                firmwareResult.updateSuccess();
            } else {
                firmwareResult.verificationFail();
            }
        } catch (IOException e) {
            e.printStackTrace();
            firmwareResult.updateFail();
        }
    }

    /**
     * 校验打印机固件
     *
     * @param bytes
     * @return
     */
    private boolean verificationBin(byte[] bytes) {
        if(bytes.length < 72)
            return false;
        //计算固件校验值
        long verificationValue = em_crc32(bytes, bytes.length - 8);
        //获取固件校验值
        long binVerificationValue = (bytes[bytes.length - 1] & 0x000000ffL) << 24 |
                (bytes[bytes.length - 2] & 0x000000ffL) << 16 |
                (bytes[bytes.length - 3] & 0x000000ffL) << 8 |
                (bytes[bytes.length - 4] & 0x000000ffL);
        return verificationValue == binVerificationValue;
    }

    /**
     * CRC校验
     *
     * @param ptr 固件数据
     * @param len 固件长度
     * @return
     */
    public long em_crc32(byte[] ptr, int len) {
        long crc = 0xffffffffL;
        long da;
        for (int i = 0; i < len; i++) {
            da = crc ^ ptr[i];
            crc >>= 4;
            crc ^= crcTbl[(int) (da & 0x0f)];
            da = crc ^ (ptr[i] >> 4);
            crc >>= 4;
            crc ^= crcTbl[(int) (da & 0x0f)];
        }
        crc = ~crc;
        crc &= 0xffffffffL;
        return crc;
    }

    /**
     * Author: 猿史森林
     * Date：2017/8/26 16:03
     * 修改日期：2017/8/26 16:03
     * <p>
     * 方法修改说明：
     * <p>
     * 方法说明：将byte[]分成counts个byte[]并放在list中
     */
    private List<byte[]> getListByteArray(byte[] bytes, int counts) {
        List<byte[]> lists = new ArrayList<>();
        int f = bytes.length / counts;
        int length = 0;
        for (int i = 0; i < f; i++) {
            byte[] bbb = new byte[counts];
            for (int j = 0; j < counts; j++) {
                bbb[j] = bytes[j + i * counts];
            }
            length += bbb.length;
            lists.add(bbb);
        }
        if (length < bytes.length) {
            byte[] a = new byte[bytes.length - length];
            for (int i = 0; i < bytes.length - length; i++) {
                a[i] = bytes[length + i];
            }
            lists.add(a);
        }
        return lists;
    }

    /**
     * 读取更新的bin文件
     *
     * @param inputStream bin文件
     */
    private byte[] convertFileToByte(InputStream inputStream) {
        List<Byte> list = new ArrayList<>();
        try {
            int len = 0;
            byte[] buffer = new byte[1024];
            while ((len = inputStream.read(buffer)) != -1) {
                for (int i = 0; i < len; i++) {
                    list.add(buffer[i]);
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        int size = list.size();
        return convertByteToByte(list.toArray(new Byte[size]));
    }

    /**
     * 读取更新的bin文件
     */
    public byte[] convertFileToByte(File file) {
        List<Byte> list = new ArrayList<>();
        FileInputStream fileInputStream = null;
        try {
            fileInputStream = new FileInputStream(file);
            int len = 0;
            byte[] buffer = new byte[1024];
            while ((len = fileInputStream.read(buffer)) != -1) {
                for (int i = 0; i < len; i++) {
                    list.add(buffer[i]);
                }
            }
            fileInputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        int size = list.size();
        return convertByteToByte(list.toArray(new Byte[size]));
    }

    /**
     * 将Byte数组转换成byte数组
     *
     * @param data
     * @return
     */
    private byte[] convertByteToByte(Byte[] data) {
        byte[] b = new byte[data.length];
        for (int i = 0; i < data.length; i++) {
            b[i] = data[i];
        }
        return b;
    }

    public byte[] convertIntToByteArrays(int v) {
        return intToBytes(v);
    }

    private byte[] intToBytes(int dec) {
        int hex = (dec & 0xFF) << 24 | ((dec >> 8) & 0xFF) << 16 | ((dec >> 16) & 0xFF) << 8 | ((dec >> 24) & 0xFF);

        byte hexs[] = new byte[4];
        hexs[3] = (byte) (hex & 0xFF);
        hexs[2] = (byte) ((hex >> 8) & 0xFF);
        hexs[1] = (byte) ((hex >> 16) & 0xFF);
        hexs[0] = (byte) ((hex >> 24) & 0xFF);
        return hexs;
    }

    public byte[] mergeArrays(byte[] a, byte[] b) {
        byte[] bytes = new byte[a.length + b.length];
        System.arraycopy(a, 0, bytes, 0, a.length);
        System.arraycopy(b, 0, bytes, a.length, b.length);
        return bytes;
    }

    public static byte[] mergeArrays(byte[] bytes1, byte[] bytes2, byte[] bytes3) {
        byte[] bytes = new byte[bytes1.length + bytes2.length + bytes3.length];
        System.arraycopy(bytes1, 0, bytes, 0, bytes1.length);
        System.arraycopy(bytes2, 0, bytes, bytes1.length, bytes2.length);
        System.arraycopy(bytes3, 0, bytes, bytes1.length + bytes2.length, bytes3.length);
        return bytes;
    }
}
